C      *****************************************************************
C      * Subroutine FODecay                                            *
C      * Updates KTs, for the passed layer, for calcium, magnesium and *
C      * potassium assuming a first order decay.  If an element is not *
C      * present then the decay calculations for that element are      *
C      * skipped.                                                      *
C      *                                                               *
C      * Variables:                                                    *
C      * Ca_in_OM  -In/Out- The amount of calcium in organic matter.   *
C      *                    Units are g Ca.                            *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Ca_Min_Rat-Input - The mineralization rate for calcium.       *
C      *                    Units are g Ca/(g Ca timestep).            *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * CaNum     -Local - Index for calcium.  Unitless.              *
C      * GCaPerMole-Local - The number of grams of calcium per mole.   *
C      *                    Units are g Ca/mol Ca.                     *
C      * GKPerMole -Local - The number of grams of potassium per mole. *
C      *                     (g K/mol K)                               *
C      * GMgPerMole-Local - The number of grams of magnesium per mole. *
C      *                    Units are g Mg/mol Mg.                     *
C      * K_in_OM   -In/Out- The amount of potassium in organic matter. *
C      *                    Units are g K.                             *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * K_Min_Rate-Input - The mineralization rate for potassium.     *
C      *                    Units are g K/(g K timestep).              *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * KNum      -Local - Index for potassium.  Unitless.            *
C      * LfNut     -Input - A vector, of size NComp, of the amount of  *
C      *                    each nutrient in the leaves that senesced  *
C      *                    during the day.  Units are g nutrients.    *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Lyr       -Input - Layer array index.  Unitless.              *
C      * Mg_in_OM  -In/Out- The amount of magnesium in organic matter. *
C      *                    Units are g Mg.                            *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Mg_Min_Rat-Input - The mineralization rate for magnesium.     *
C      *                    Units are g Mg/(g Mg timestep).            *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * MgNum     -Local - Index for magnesium.  Unitless.            *
C      * MinCa     -Local - The amount of calcium mineralized.         *
C      * MinK      -Local - The amount of potassium mineralized.  Units*
C      *                    are g K.                                   *
C      * MinMg     -Local - The amount of magnesium mineralized.  Units*
C      *                    are g Mg.                                  *
C      * Name      -Local - Used for passing data to WhereIs function. *
C      *                    Unitless.                                  *
C      * NLayers   -Input - The number of soil layers in the system.   *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Unitless.         *
C      *                    (Common block VPos, file VModel.f)         *
C      * RtNut     -Input - A matrix, NLayers by NComps, of the        *
C      *                    amount of nutrients in the roots, in each  *
C      *                    layer, that senesced during the day.  Input*
C      *                    from TREGRO, or read into the model by the *
C      *                    routine InParam.  Units are g nutr.        *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * WhereIs   -Fcn   - Returns the index of requested species. If *
C      *                    the species is not found a -1 is returned. *
C      *                    Unitless.                                  *
C      *****************************************************************
       SUBROUTINE FODECAY(LYR)
       IMPLICIT NONE
       INCLUDE 'VMODEL.INC'
       INTEGER CANUM, KNUM, LYR, MGNUM, WHEREIS
       REAL*8 GCAPERMOLE, GKPERMOLE, GMGPERMOLE, MINCA, MINK, MINMG
							CHARACTER*8 NAME
       PARAMETER (GCAPERMOLE = 40.0, GKPERMOLE = 39.0)
       PARAMETER (GMGPERMOLE = 24.3)

C      *------------------*
C      * Get the indices. *
C      *------------------*
       NAME = 'Ca'
       CANUM = WHEREIS(NAME)
       NAME = 'Mg'
       MGNUM = WHEREIS(NAME)
       NAME = 'K'
       KNUM = WHEREIS(NAME)

C      *------------------------*
C      * Update KT for calcium. *
C      *------------------------*
       IF (CANUM.NE.-1) THEN

C         *-------------------------------------------*
C         * Calculate the amount mineralized in g Ca. *
C         *-------------------------------------------*
          MINCA = CA_MIN_RAT * CA_IN_OM(LYR)

C         *---------------------------------------*
C         * Update the KT for calcium (moles Ca). *
C         *---------------------------------------*
          KT(CANUM,LYR) = KT(CANUM,LYR) + (MINCA / GCAPERMOLE)

C         *------------------------------------------------------------*
C         * Update the amount of calcium in organic matter (grams Ca). *
C         *------------------------------------------------------------*
          IF (LYR.EQ.1) THEN
             CA_IN_OM(LYR) = MAX(0.0, CA_IN_OM(LYR) - MINCA + 
     >                       RTNUT(CANUM,LYR) + LFNUT(CANUM))
          ELSE
             CA_IN_OM(LYR) = MAX(0.0, CA_IN_OM(LYR) - MINCA + 
     >                       RTNUT(CANUM,LYR))
          END IF
          IF (CA_IN_OM(LYR).LT.1E-20) CA_IN_OM(LYR) = 0.0

       ENDIF

C      *--------------------------*
C      * Update KT for magnesium. *
C      *--------------------------*
       IF (MGNUM.NE.-1) THEN

C         *-------------------------------------------*
C         * Calculate the amount mineralized in g Mg. *
C         *-------------------------------------------*
          MINMG = MG_MIN_RAT * MG_IN_OM(LYR)

C         *-----------------------------------------*
C         * Update the KT for magnesium (moles Mg). *
C         *-----------------------------------------*
          KT(MGNUM,LYR) = KT(MGNUM,LYR) + (MINMG / GMGPERMOLE)

C         *---------------------------------------------------------------*
C         * Update the amount of magnesium in organic matter (grams Mg).  *
C         *---------------------------------------------------------------*
          IF (LYR.EQ.1) THEN
             MG_IN_OM(LYR) = MAX(0.0, MG_IN_OM(LYR) - MINMG + 
     >                       RTNUT(MGNUM,LYR) + LFNUT(MGNUM))
          ELSE
             MG_IN_OM(LYR) = MAX(0.0, MG_IN_OM(LYR) - MINMG + 
     >                       RTNUT(MGNUM,LYR))
          END IF
          IF (MG_IN_OM(LYR).LT.1E-20) MG_IN_OM(LYR) = 0.0

       ENDIF

C      *--------------------------*
C      * Update KT for potassium. *
C      *--------------------------*
       IF (KNUM.NE.-1) THEN

C         *-------------------------------------------*
C         * Calculate the amount mineralized in g K.  *
C         *-------------------------------------------*
          MINK = K_MIN_RATE * K_IN_OM(LYR)

C         *-----------------------------------------*
C         * Update the KT for potassium (moles K).  *
C         *-----------------------------------------*
          KT(KNUM,LYR) = KT(KNUM,LYR) + (MINK / GKPERMOLE)

C         *-------------------------------------------------------------*
C         * Update the amount of potassium in organic matter (grams K). *
C         *-------------------------------------------------------------*
          IF (LYR.EQ.1) THEN
             K_IN_OM(LYR) = MAX(0.0, K_IN_OM(LYR) - MINK + 
     >                      RTNUT(KNUM,LYR) + LFNUT(KNUM))
          ELSE
             K_IN_OM(LYR) = MAX(0.0, K_IN_OM(LYR) - MINK + 
     >                      RTNUT(KNUM,LYR))
          END IF
          IF (K_IN_OM(LYR).LT.1E-20) K_IN_OM(LYR) = 0.0

       ENDIF

	      RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
